import { NextResponse } from "next/server";
import nodemailer from "nodemailer";

// Service labels for better email formatting
const serviceLabels: Record<string, string> = {
  "small-business": "Small Business Website",
  "custom-dev": "Custom Web Development",
  hosting: "Hosting & Maintenance",
  "not-sure": "Not Sure / Other",
};

const budgetLabels: Record<string, string> = {
  "under-1000": "Under $1,000",
  "1000-2500": "$1,000 - $2,500",
  "2500-5000": "$2,500 - $5,000",
  "5000-10000": "$5,000 - $10,000",
  "over-10000": "$10,000+",
};

export async function POST(request: Request) {
  try {
    const body = await request.json();
    const { firstName, lastName, email, company, service, budget, message } =
      body;

    // Validate required fields
    if (!firstName || !lastName || !email || !service || !message) {
      return NextResponse.json(
        { error: "Please fill in all required fields" },
        { status: 400 }
      );
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { error: "Please enter a valid email address" },
        { status: 400 }
      );
    }

    // Create transporter for mail.traunico.com
    const transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST || "mail.traunico.com",
      port: parseInt(process.env.SMTP_PORT || "465"),
      secure: process.env.SMTP_SECURE === "true", // true for port 465 (SSL)
      auth: {
        user: process.env.SMTP_USER || "team@traunico.com",
        pass: process.env.SMTP_PASS || "",
      },
      tls: {
        // Allow self-signed certificates if needed
        rejectUnauthorized: process.env.NODE_ENV === "production",
      },
    });

    // Format the email content
    const serviceName = serviceLabels[service] || service;
    const budgetRange = budget
      ? budgetLabels[budget] || budget
      : "Not specified";

    const emailContent = `
New Project Inquiry from Traunico Website

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

CONTACT INFORMATION
━━━━━━━━━━━━━━━━━━━
Name: ${firstName} ${lastName}
Email: ${email}
Business: ${company || "Not provided"}

PROJECT DETAILS
━━━━━━━━━━━━━━━
Service: ${serviceName}
Budget: ${budgetRange}

MESSAGE
━━━━━━━
${message}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
This inquiry was submitted through the Traunico website contact form.
    `.trim();

    const htmlContent = `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; line-height: 1.6; color: #1f2933; }
    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
    .header { background: linear-gradient(135deg, #0f766e 0%, #0d5f59 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
    .header h1 { margin: 0; font-size: 24px; }
    .content { background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; border-top: none; border-radius: 0 0 8px 8px; }
    .section { margin-bottom: 24px; }
    .section-title { font-size: 14px; font-weight: 600; color: #0f766e; text-transform: uppercase; letter-spacing: 0.05em; margin-bottom: 12px; border-bottom: 2px solid #0f766e; padding-bottom: 8px; }
    .field { margin-bottom: 8px; }
    .field-label { font-weight: 600; color: #52606d; }
    .field-value { color: #1f2933; }
    .message-box { background: white; padding: 16px; border-radius: 8px; border: 1px solid #e5e7eb; white-space: pre-wrap; }
    .footer { text-align: center; margin-top: 24px; font-size: 12px; color: #7b8794; }
    .badge { display: inline-block; background: #6d28d9; color: white; padding: 4px 12px; border-radius: 50px; font-size: 12px; font-weight: 600; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <h1>🚀 New Project Inquiry</h1>
    </div>
    <div class="content">
      <div class="section">
        <div class="section-title">Contact Information</div>
        <div class="field">
          <span class="field-label">Name:</span>
          <span class="field-value">${firstName} ${lastName}</span>
        </div>
        <div class="field">
          <span class="field-label">Email:</span>
          <span class="field-value"><a href="mailto:${email}">${email}</a></span>
        </div>
        <div class="field">
          <span class="field-label">Business:</span>
          <span class="field-value">${company || "Not provided"}</span>
        </div>
      </div>
      
      <div class="section">
        <div class="section-title">Project Details</div>
        <div class="field">
          <span class="field-label">Service:</span>
          <span class="badge">${serviceName}</span>
        </div>
        <div class="field">
          <span class="field-label">Budget:</span>
          <span class="field-value">${budgetRange}</span>
        </div>
      </div>
      
      <div class="section">
        <div class="section-title">Message</div>
        <div class="message-box">${message.replace(/\n/g, "<br>")}</div>
      </div>
      
      <div class="footer">
        This inquiry was submitted through the Traunico website contact form.
      </div>
    </div>
  </div>
</body>
</html>
    `.trim();

    // Send email
    await transporter.sendMail({
      from: `"Traunico Website" <${
        process.env.SMTP_FROM || "noreply@traunico.com"
      }>`,
      to: "team@traunico.com",
      replyTo: email,
      subject: `New Project Inquiry: ${serviceName} - ${firstName} ${lastName}`,
      text: emailContent,
      html: htmlContent,
    });

    return NextResponse.json(
      { message: "Email sent successfully" },
      { status: 200 }
    );
  } catch (error) {
    console.error("Error sending email:", error);
    return NextResponse.json(
      { error: "Failed to send email. Please try again later." },
      { status: 500 }
    );
  }
}
