"use client";

import { useState, useEffect, Suspense } from "react";
import { useSearchParams } from "next/navigation";

const faqs = [
  {
    question: "How long does a typical project take?",
    answer:
      "Small business websites typically take 2-4 weeks from start to launch. Custom development projects vary based on complexity, but we'll provide a clear timeline in your proposal.",
  },
  {
    question: "Do I need to provide content and images?",
    answer:
      "We can work with content you provide, or help guide you through the process. We recommend having some ideas about what you want to say, but we're here to help refine your messaging.",
  },
  {
    question: "What if I need changes after launch?",
    answer:
      "Our hosting & maintenance plans include minor content updates. For larger changes, we offer hourly development support or can scope a new project.",
  },
  {
    question: "Do you work with clients outside your area?",
    answer:
      "Absolutely! We work with clients worldwide. All our communication happens via video calls, email, and project management tools.",
  },
];

function ContactForm() {
  const searchParams = useSearchParams();
  const [formData, setFormData] = useState({
    firstName: "",
    lastName: "",
    email: "",
    company: "",
    service: "",
    budget: "",
    message: "",
  });
  const [status, setStatus] = useState<"idle" | "loading" | "success" | "error">(
    "idle"
  );
  const [errorMessage, setErrorMessage] = useState("");

  useEffect(() => {
    const serviceParam = searchParams.get("service");
    if (serviceParam) {
      setFormData((prev) => ({ ...prev, service: serviceParam }));
    }
  }, [searchParams]);

  const handleChange = (
    e: React.ChangeEvent<
      HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement
    >
  ) => {
    setFormData((prev) => ({
      ...prev,
      [e.target.name]: e.target.value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setStatus("loading");
    setErrorMessage("");

    try {
      const response = await fetch("/api/contact", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(formData),
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || "Something went wrong");
      }

      setStatus("success");
      setFormData({
        firstName: "",
        lastName: "",
        email: "",
        company: "",
        service: "",
        budget: "",
        message: "",
      });

      // Reset status after 5 seconds
      setTimeout(() => setStatus("idle"), 5000);
    } catch (error) {
      setStatus("error");
      setErrorMessage(
        error instanceof Error ? error.message : "Something went wrong"
      );
    }
  };

  return (
    <form
      onSubmit={handleSubmit}
      className="bg-white rounded-[var(--radius-lg)] p-[var(--space-xl)] shadow-lg"
    >
      <h3 className="text-[1.5rem] font-semibold mb-[var(--space-lg)]">
        Tell Us About Your Project
      </h3>

      <div className="grid grid-cols-1 sm:grid-cols-2 gap-[var(--space-md)] mb-[var(--space-md)]">
        <div>
          <label
            htmlFor="firstName"
            className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
          >
            First Name *
          </label>
          <input
            type="text"
            id="firstName"
            name="firstName"
            required
            value={formData.firstName}
            onChange={handleChange}
            placeholder="John"
            className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)]"
          />
        </div>
        <div>
          <label
            htmlFor="lastName"
            className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
          >
            Last Name *
          </label>
          <input
            type="text"
            id="lastName"
            name="lastName"
            required
            value={formData.lastName}
            onChange={handleChange}
            placeholder="Smith"
            className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)]"
          />
        </div>
      </div>

      <div className="mb-[var(--space-md)]">
        <label
          htmlFor="email"
          className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
        >
          Email Address *
        </label>
        <input
          type="email"
          id="email"
          name="email"
          required
          value={formData.email}
          onChange={handleChange}
          placeholder="john@example.com"
          className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)]"
        />
      </div>

      <div className="mb-[var(--space-md)]">
        <label
          htmlFor="company"
          className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
        >
          Business Name
        </label>
        <input
          type="text"
          id="company"
          name="company"
          value={formData.company}
          onChange={handleChange}
          placeholder="Your Business Name (optional)"
          className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)]"
        />
      </div>

      <div className="mb-[var(--space-md)]">
        <label
          htmlFor="service"
          className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
        >
          Service Interested In *
        </label>
        <select
          id="service"
          name="service"
          required
          value={formData.service}
          onChange={handleChange}
          className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)]"
        >
          <option value="">Select a service...</option>
          <option value="small-business">Small Business Website</option>
          <option value="custom-dev">Custom Web Development</option>
          <option value="hosting">Hosting & Maintenance</option>
          <option value="not-sure">Not Sure / Other</option>
        </select>
      </div>

      <div className="mb-[var(--space-md)]">
        <label
          htmlFor="budget"
          className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
        >
          Estimated Budget
        </label>
        <select
          id="budget"
          name="budget"
          value={formData.budget}
          onChange={handleChange}
          className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)]"
        >
          <option value="">Select a range (optional)...</option>
          <option value="under-1000">Under $1,000</option>
          <option value="1000-2500">$1,000 - $2,500</option>
          <option value="2500-5000">$2,500 - $5,000</option>
          <option value="5000-10000">$5,000 - $10,000</option>
          <option value="over-10000">$10,000+</option>
        </select>
      </div>

      <div className="mb-[var(--space-md)]">
        <label
          htmlFor="message"
          className="block font-[var(--font-heading)] text-[0.9375rem] font-medium mb-[var(--space-xs)]"
        >
          Project Details *
        </label>
        <textarea
          id="message"
          name="message"
          required
          value={formData.message}
          onChange={handleChange}
          placeholder="Tell us a bit about your project, goals, and timeline..."
          className="w-full px-4 py-3.5 font-[var(--font-body)] text-[1rem] text-[var(--color-text)] bg-[var(--color-light-gray)] border-2 border-transparent rounded-[var(--radius-md)] transition-all focus:outline-none focus:bg-white focus:border-[var(--color-primary)] min-h-[120px] resize-y"
        />
      </div>

      <button
        type="submit"
        disabled={status === "loading"}
        className="w-full inline-flex items-center justify-center gap-[var(--space-xs)] px-7 py-4 bg-[var(--color-primary)] text-white font-[var(--font-heading)] text-lg font-semibold rounded-[var(--radius-md)] shadow-md hover:bg-[var(--color-primary-dark)] hover:-translate-y-0.5 hover:shadow-lg transition-all disabled:opacity-70 disabled:cursor-not-allowed disabled:transform-none"
      >
        {status === "loading" ? (
          <>
            <svg
              width="20"
              height="20"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="2"
              className="animate-spin-custom"
            >
              <circle
                cx="12"
                cy="12"
                r="10"
                strokeDasharray="32"
                strokeDashoffset="12"
              ></circle>
            </svg>
            Sending...
          </>
        ) : status === "success" ? (
          <>
            <svg
              width="20"
              height="20"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="2"
            >
              <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            Message Sent!
          </>
        ) : (
          <>
            Send Message
            <svg
              width="20"
              height="20"
              viewBox="0 0 24 24"
              fill="none"
              stroke="currentColor"
              strokeWidth="2"
            >
              <line x1="22" y1="2" x2="11" y2="13"></line>
              <polygon points="22 2 15 22 11 13 2 9 22 2"></polygon>
            </svg>
          </>
        )}
      </button>

      {status === "error" && (
        <p className="text-red-500 text-center mt-[var(--space-md)]">
          {errorMessage || "Something went wrong. Please try again."}
        </p>
      )}

      {status === "success" && (
        <p className="text-[var(--color-primary)] text-center mt-[var(--space-md)]">
          Thank you! We&apos;ll be in touch within 24 hours.
        </p>
      )}

      <p className="text-[0.875rem] text-[var(--color-text-light)] text-center mt-[var(--space-md)]">
        We respect your privacy and will never share your information.
      </p>
    </form>
  );
}

export default function ContactPage() {
  return (
    <>
      {/* Contact Section */}
      <section className="pt-[calc(80px+var(--space-3xl))] py-[var(--space-3xl)] bg-[var(--color-light-gray)]">
        <div className="max-w-[var(--container-max)] mx-auto px-[var(--space-lg)]">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-[var(--space-2xl)] items-start">
            {/* Contact Info */}
            <div>
              <span className="inline-block font-[var(--font-heading)] text-[0.875rem] font-semibold text-[var(--color-primary)] uppercase tracking-wider mb-[var(--space-sm)]">
                Get in Touch
              </span>
              <h1 className="text-[clamp(2rem,4vw,3rem)] font-semibold mb-[var(--space-md)]">
                Let&apos;s Start Your Project
              </h1>
              <p className="text-[var(--color-text-muted)] mb-[var(--space-lg)]">
                Ready to bring your vision to life? Fill out the form and
                we&apos;ll get back to you within 24 hours to schedule a free
                consultation.
              </p>

              <ul className="space-y-[var(--space-sm)] mb-[var(--space-xl)]">
                <li className="flex items-center gap-[var(--space-md)]">
                  <svg
                    viewBox="0 0 24 24"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    className="w-6 h-6 text-[var(--color-primary)]"
                  >
                    <path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"></path>
                    <polyline points="22,6 12,13 2,6"></polyline>
                  </svg>
                  <div>
                    <strong>Email</strong>
                    <br />
                    <a
                      href="mailto:team@traunico.com"
                      className="text-[var(--color-primary)] hover:text-[var(--color-secondary)]"
                    >
                      team@traunico.com
                    </a>
                  </div>
                </li>
                <li className="flex items-center gap-[var(--space-md)]">
                  <svg
                    viewBox="0 0 24 24"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    className="w-6 h-6 text-[var(--color-primary)]"
                  >
                    <circle cx="12" cy="12" r="10"></circle>
                    <polyline points="12 6 12 12 16 14"></polyline>
                  </svg>
                  <div>
                    <strong>Response Time</strong>
                    <br />
                    Within 24 hours
                  </div>
                </li>
                <li className="flex items-center gap-[var(--space-md)]">
                  <svg
                    viewBox="0 0 24 24"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    className="w-6 h-6 text-[var(--color-primary)]"
                  >
                    <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path>
                    <circle cx="12" cy="10" r="3"></circle>
                  </svg>
                  <div>
                    <strong>Location</strong>
                    <br />
                    Remote & Available Worldwide
                  </div>
                </li>
              </ul>

              <div className="pt-[var(--space-lg)] border-t border-[var(--color-border)]">
                <h4 className="text-[1rem] font-semibold mb-[var(--space-md)]">
                  What Happens Next?
                </h4>
                <ol className="space-y-[var(--space-sm)]">
                  {[
                    "We review your project details",
                    "Schedule a free discovery call",
                    "Receive a custom proposal",
                  ].map((step, index) => (
                    <li
                      key={index}
                      className="flex gap-[var(--space-sm)] text-[var(--color-text-muted)]"
                    >
                      <span className="flex items-center justify-center w-6 h-6 bg-[var(--color-primary)] text-white rounded-full text-[0.75rem] font-semibold flex-shrink-0">
                        {index + 1}
                      </span>
                      {step}
                    </li>
                  ))}
                </ol>
              </div>
            </div>

            {/* Contact Form */}
            <Suspense
              fallback={
                <div className="bg-white rounded-[var(--radius-lg)] p-[var(--space-xl)] shadow-lg animate-pulse">
                  <div className="h-8 bg-gray-200 rounded mb-6 w-2/3"></div>
                  <div className="space-y-4">
                    <div className="h-12 bg-gray-200 rounded"></div>
                    <div className="h-12 bg-gray-200 rounded"></div>
                    <div className="h-12 bg-gray-200 rounded"></div>
                    <div className="h-24 bg-gray-200 rounded"></div>
                    <div className="h-14 bg-gray-200 rounded"></div>
                  </div>
                </div>
              }
            >
              <ContactForm />
            </Suspense>
          </div>
        </div>
      </section>

      {/* FAQ Section */}
      <section className="py-[var(--space-3xl)] bg-white">
        <div className="max-w-[var(--container-narrow)] mx-auto px-[var(--space-lg)]">
          <div className="text-center mb-[var(--space-2xl)]">
            <span className="inline-block font-[var(--font-heading)] text-[0.875rem] font-semibold text-[var(--color-primary)] uppercase tracking-wider mb-[var(--space-sm)]">
              Questions?
            </span>
            <h2 className="text-[clamp(2rem,4vw,3rem)] font-semibold">
              Frequently Asked
            </h2>
          </div>

          <div className="space-y-[var(--space-md)]">
            {faqs.map((faq, index) => (
              <div
                key={index}
                className="bg-[var(--color-light-gray)] rounded-[var(--radius-md)] p-[var(--space-lg)]"
              >
                <h4 className="text-[1.125rem] font-semibold text-[var(--color-text)] mb-[var(--space-sm)]">
                  {faq.question}
                </h4>
                <p className="text-[var(--color-text-muted)]">{faq.answer}</p>
              </div>
            ))}
          </div>
        </div>
      </section>
    </>
  );
}

