import Link from "next/link";
import { Metadata } from "next";

export const metadata: Metadata = {
  title: "Services – Traunico Web Development",
  description:
    "Explore Traunico's web development services: Small business websites, custom web development, and hosting & maintenance packages.",
};

const CheckIcon = () => (
  <svg
    width="20"
    height="20"
    viewBox="0 0 24 24"
    fill="none"
    stroke="var(--color-primary)"
    strokeWidth="2"
    className="flex-shrink-0 mt-0.5"
  >
    <polyline points="20 6 9 17 4 12"></polyline>
  </svg>
);

const services = [
  {
    id: "small-business",
    label: "Perfect for Local Businesses",
    title: "Small Business Websites",
    description:
      "Get a clean, professional website that makes a great first impression. Built on platforms you can manage yourself, with all the features your business needs to attract and convert customers.",
    features: [
      "Built on Wix, Squarespace, or WordPress",
      "Fully mobile-responsive design",
      "Contact forms & basic SEO setup",
      "Up to 5 pages included",
      "Training on how to update content",
    ],
    price: "Starting at $800",
    cta: "Get Started",
    icon: (
      <svg
        width="120"
        height="120"
        viewBox="0 0 24 24"
        fill="none"
        stroke="currentColor"
        strokeWidth="1.5"
        strokeLinecap="round"
        strokeLinejoin="round"
      >
        <rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect>
        <line x1="3" y1="9" x2="21" y2="9"></line>
        <line x1="9" y1="21" x2="9" y2="9"></line>
      </svg>
    ),
    iconLabel: "Professional Website",
  },
  {
    id: "custom-dev",
    label: "For Advanced Needs",
    title: "Custom Web Development",
    description:
      "When off-the-shelf solutions won't cut it, we build custom web applications tailored to your specific requirements. From interactive dashboards to complex integrations, we deliver solutions that scale.",
    features: [
      "Built with Next.js & modern technologies",
      "Custom databases & API integrations",
      "User authentication & admin panels",
      "Interactive dashboards & data visualization",
      "Third-party service integrations",
    ],
    price: "Quote-based pricing",
    cta: "Discuss Your Project",
    icon: (
      <svg
        width="120"
        height="120"
        viewBox="0 0 24 24"
        fill="none"
        stroke="currentColor"
        strokeWidth="1.5"
        strokeLinecap="round"
        strokeLinejoin="round"
      >
        <polyline points="16 18 22 12 16 6"></polyline>
        <polyline points="8 6 2 12 8 18"></polyline>
      </svg>
    ),
    iconLabel: "Custom Development",
    reverse: true,
  },
  {
    id: "hosting",
    label: "Ongoing Support",
    title: "Hosting & Maintenance",
    description:
      "Keep your website running smoothly with our managed hosting and maintenance packages. We handle the technical details so you can focus on your business.",
    features: [
      "Reliable, managed hosting",
      "Regular security updates",
      "Automated backups",
      "Minor content edits included",
      "Monthly performance reports",
    ],
    price: "Monthly & yearly plans available",
    cta: "View Plans",
    icon: (
      <svg
        width="120"
        height="120"
        viewBox="0 0 24 24"
        fill="none"
        stroke="currentColor"
        strokeWidth="1.5"
        strokeLinecap="round"
        strokeLinejoin="round"
      >
        <rect x="2" y="2" width="20" height="8" rx="2" ry="2"></rect>
        <rect x="2" y="14" width="20" height="8" rx="2" ry="2"></rect>
        <line x1="6" y1="6" x2="6.01" y2="6"></line>
        <line x1="6" y1="18" x2="6.01" y2="18"></line>
      </svg>
    ),
    iconLabel: "Managed Hosting",
  },
];

const steps = [
  {
    number: 1,
    title: "Discovery Call",
    description:
      "We start with a free consultation to understand your business, goals, and requirements.",
  },
  {
    number: 2,
    title: "Custom Proposal",
    description:
      "You'll receive a detailed proposal with scope, timeline, and transparent pricing.",
  },
  {
    number: 3,
    title: "Design & Build",
    description:
      "We design and build your site, keeping you involved with regular updates and feedback sessions.",
  },
  {
    number: 4,
    title: "Launch & Support",
    description:
      "After thorough testing, we launch your site and provide training and ongoing support.",
  },
];

const pricingPlans = [
  {
    title: "Small Business Website",
    description: "Perfect for getting your business online",
    price: "$800",
    priceLabel: "starting",
    note: "One-time project fee",
    features: [
      "Up to 5 pages",
      "Mobile-responsive design",
      "Contact form setup",
      "Basic SEO optimization",
      "Training included",
    ],
    cta: "Get Started",
    href: "/contact?service=small-business",
    featured: false,
  },
  {
    title: "Custom Development",
    description: "For advanced web applications",
    price: "Custom",
    priceLabel: "quote",
    note: "Based on project scope",
    features: [
      "Full-stack development",
      "Custom functionality",
      "Database & API integration",
      "Admin dashboards",
      "Scalable architecture",
    ],
    cta: "Request Quote",
    href: "/contact?service=custom-dev",
    featured: true,
  },
  {
    title: "Hosting & Maintenance",
    description: "Keep your site running smoothly",
    price: "$50",
    priceLabel: "/month",
    note: "Or $500/year (save $100)",
    features: [
      "Managed hosting",
      "Security updates",
      "Daily backups",
      "Minor content edits",
      "Priority support",
    ],
    cta: "Learn More",
    href: "/contact?service=hosting",
    featured: false,
  },
];

export default function ServicesPage() {
  return (
    <>
      {/* Hero Section */}
      <section className="pt-[calc(80px+var(--space-3xl))] pb-[var(--space-2xl)] bg-gradient-to-br from-white to-[var(--color-light-gray)] relative overflow-hidden">
        <div className="max-w-[var(--container-max)] mx-auto px-[var(--space-lg)]">
          <div className="text-center max-w-[800px] mx-auto">
            <h1 className="animate-in text-[clamp(2.5rem,5vw,4rem)] font-bold leading-[1.1] mb-[var(--space-md)]">
              Web Solutions That{" "}
              <span className="text-[var(--color-primary)]">Work for You</span>
            </h1>
            <p className="animate-in animate-delay-1 text-[1.25rem] text-[var(--color-text-muted)] mb-[var(--space-xl)] mx-auto">
              From simple business websites to custom applications, we offer
              services designed to help your business thrive online.
            </p>
            <div className="animate-in animate-delay-2 flex flex-wrap justify-center gap-[var(--space-md)]">
              <a
                href="#services"
                className="inline-flex items-center justify-center gap-[var(--space-xs)] px-7 py-4 bg-[var(--color-primary)] text-white font-[var(--font-heading)] text-lg font-semibold rounded-[var(--radius-md)] shadow-md hover:bg-[var(--color-primary-dark)] hover:-translate-y-0.5 hover:shadow-lg transition-all"
              >
                Explore Services
              </a>
              <Link
                href="/contact"
                className="inline-flex items-center justify-center gap-[var(--space-xs)] px-7 py-4 bg-transparent text-[var(--color-primary)] font-[var(--font-heading)] text-lg font-semibold rounded-[var(--radius-md)] border-2 border-[var(--color-primary)] hover:bg-[var(--color-primary)] hover:text-white transition-all"
              >
                Get a Quote
              </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Services Detail Section */}
      <section id="services" className="py-[var(--space-2xl)] bg-white">
        <div className="max-w-[var(--container-max)] mx-auto px-[var(--space-lg)]">
          {services.map((service, index) => (
            <div
              key={service.id}
              id={service.id}
              className={`mb-[var(--space-3xl)] last:mb-0 scroll-mt-24`}
            >
              <div
                className={`grid grid-cols-1 lg:grid-cols-2 gap-[var(--space-2xl)] items-center ${
                  service.reverse ? "lg:[&>*:first-child]:order-2" : ""
                }`}
              >
                <div>
                  <span className="inline-block font-[var(--font-heading)] text-[0.875rem] font-semibold text-[var(--color-primary)] uppercase tracking-wider mb-[var(--space-sm)]">
                    {service.label}
                  </span>
                  <h2 className="text-[clamp(2rem,4vw,3rem)] font-semibold mb-[var(--space-md)]">
                    {service.title}
                  </h2>
                  <p className="text-[var(--color-text-muted)] mb-[var(--space-lg)]">
                    {service.description}
                  </p>

                  <ul className="space-y-[var(--space-sm)] mb-[var(--space-lg)]">
                    {service.features.map((feature, idx) => (
                      <li key={idx} className="flex items-start gap-[var(--space-sm)]">
                        <CheckIcon />
                        <span>{feature}</span>
                      </li>
                    ))}
                  </ul>

                  <p className="font-[var(--font-heading)] font-semibold text-[var(--color-secondary)] mb-[var(--space-md)]">
                    {service.price}
                  </p>

                  <Link
                    href={`/contact?service=${service.id}`}
                    className="inline-flex items-center justify-center gap-[var(--space-xs)] px-6 py-3 bg-[var(--color-primary)] text-white font-[var(--font-heading)] font-semibold rounded-[var(--radius-md)] shadow-md hover:bg-[var(--color-primary-dark)] hover:-translate-y-0.5 hover:shadow-lg transition-all"
                  >
                    {service.cta}
                  </Link>
                </div>

                <div className="bg-[var(--color-light-gray)] rounded-[var(--radius-lg)] p-[var(--space-xl)] flex items-center justify-center min-h-[320px]">
                  <div className="text-center text-[var(--color-primary)]">
                    {service.icon}
                    <p className="mt-[var(--space-md)] font-[var(--font-heading)] font-semibold">
                      {service.iconLabel}
                    </p>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </section>

      {/* How It Works */}
      <section className="py-[var(--space-3xl)] bg-[var(--color-light-gray)]">
        <div className="max-w-[var(--container-max)] mx-auto px-[var(--space-lg)]">
          <div className="text-center max-w-[640px] mx-auto mb-[var(--space-2xl)]">
            <span className="inline-block font-[var(--font-heading)] text-[0.875rem] font-semibold text-[var(--color-primary)] uppercase tracking-wider mb-[var(--space-sm)]">
              Simple Process
            </span>
            <h2 className="text-[clamp(2rem,4vw,3rem)] font-semibold mb-[var(--space-md)]">
              How We Work Together
            </h2>
            <p className="text-[1.125rem] text-[var(--color-text-muted)]">
              From first conversation to launch day, here&apos;s what you can
              expect.
            </p>
          </div>

          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-[var(--space-lg)]">
            {steps.map((step) => (
              <div key={step.number} className="text-center relative">
                <div className="w-[72px] h-[72px] flex items-center justify-center bg-white border-2 border-[var(--color-primary)] text-[var(--color-primary)] font-[var(--font-heading)] text-2xl font-bold rounded-full mx-auto mb-[var(--space-md)] relative z-10">
                  {step.number}
                </div>
                <h4 className="text-[1.125rem] font-semibold mb-[var(--space-xs)]">
                  {step.title}
                </h4>
                <p className="text-[1rem] text-[var(--color-text-muted)]">
                  {step.description}
                </p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Pricing Preview */}
      <section id="pricing" className="py-[var(--space-3xl)] bg-white scroll-mt-24">
        <div className="max-w-[var(--container-max)] mx-auto px-[var(--space-lg)]">
          <div className="text-center max-w-[640px] mx-auto mb-[var(--space-2xl)]">
            <span className="inline-block font-[var(--font-heading)] text-[0.875rem] font-semibold text-[var(--color-primary)] uppercase tracking-wider mb-[var(--space-sm)]">
              Investment
            </span>
            <h2 className="text-[clamp(2rem,4vw,3rem)] font-semibold mb-[var(--space-md)]">
              Pricing Overview
            </h2>
            <p className="text-[1.125rem] text-[var(--color-text-muted)]">
              Transparent pricing with no hidden fees. Every project is unique,
              so contact us for a personalized quote.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-[var(--space-lg)]">
            {pricingPlans.map((plan, index) => (
              <div
                key={index}
                className={`bg-white border rounded-[var(--radius-lg)] p-[var(--space-xl)] text-center transition-all hover:shadow-xl hover:-translate-y-1 relative ${
                  plan.featured
                    ? "border-[var(--color-primary)] shadow-lg"
                    : "border-[var(--color-border)]"
                }`}
              >
                {plan.featured && (
                  <span className="absolute -top-3 left-1/2 -translate-x-1/2 bg-[var(--color-secondary)] text-white font-[var(--font-heading)] text-[0.75rem] font-semibold px-4 py-1.5 rounded-full uppercase tracking-wider">
                    Most Popular
                  </span>
                )}
                <h3 className="text-[1.5rem] font-semibold mb-[var(--space-sm)]">
                  {plan.title}
                </h3>
                <p className="text-[var(--color-text-muted)] text-[1rem] mb-[var(--space-md)]">
                  {plan.description}
                </p>
                <p className="font-[var(--font-heading)] text-[2.5rem] font-bold text-[var(--color-text)] mb-[var(--space-xs)]">
                  {plan.price}{" "}
                  <span className="text-[1rem] font-medium text-[var(--color-text-muted)]">
                    {plan.priceLabel}
                  </span>
                </p>
                <p className="text-[0.9375rem] text-[var(--color-text-light)] mb-[var(--space-lg)]">
                  {plan.note}
                </p>
                <ul className="text-left mb-[var(--space-lg)] space-y-[var(--space-xs)]">
                  {plan.features.map((feature, idx) => (
                    <li
                      key={idx}
                      className="flex items-start gap-[var(--space-sm)] text-[0.9375rem] text-[var(--color-text-muted)] py-[var(--space-xs)]"
                    >
                      <CheckIcon />
                      {feature}
                    </li>
                  ))}
                </ul>
                <Link
                  href={plan.href}
                  className={`w-full inline-flex items-center justify-center gap-[var(--space-xs)] px-6 py-3 font-[var(--font-heading)] font-semibold rounded-[var(--radius-md)] transition-all ${
                    plan.featured
                      ? "bg-[var(--color-primary)] text-white shadow-md hover:bg-[var(--color-primary-dark)] hover:-translate-y-0.5 hover:shadow-lg"
                      : "bg-transparent text-[var(--color-primary)] border-2 border-[var(--color-primary)] hover:bg-[var(--color-primary)] hover:text-white"
                  }`}
                >
                  {plan.cta}
                </Link>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-[var(--space-3xl)] bg-gradient-to-br from-[var(--color-primary)] to-[var(--color-primary-dark)] text-white text-center">
        <div className="max-w-[var(--container-narrow)] mx-auto px-[var(--space-lg)]">
          <h2 className="text-[clamp(2rem,4vw,3rem)] font-semibold text-white mb-[var(--space-md)]">
            Ready to Start Your Project?
          </h2>
          <p className="text-[1.25rem] text-white/90 max-w-[560px] mx-auto mb-[var(--space-xl)]">
            Let&apos;s discuss your needs and create something great together.
          </p>
          <Link
            href="/contact"
            className="inline-flex items-center justify-center gap-[var(--space-xs)] px-7 py-4 bg-white text-[var(--color-primary)] font-[var(--font-heading)] text-lg font-semibold rounded-[var(--radius-md)] shadow-md hover:bg-[var(--color-light-gray)] hover:-translate-y-0.5 hover:shadow-lg transition-all"
          >
            Contact Us Today
          </Link>
        </div>
      </section>
    </>
  );
}

